'use strict';
const fs = require('fs');
const convert = require('xml-js');
const NginxConfFile = require('nginx-conf').NginxConfFile;

let DBS_CONFIG_PATH = '/opt/FileMaker/FileMaker Server/Data/Preferences/dbs_config.xml';
let NGINX_CONF_PATH = '/opt/FileMaker/FileMaker Server/Tools/NginxLB/fms_LB_HTTPS.template.conf';
let NGINX_CONF_OUTPUT_PATH = '/opt/FileMaker/FileMaker Server/Tools/NginxLB/fms_LB_HTTPS.user.conf';
const HTTPS_PORT = ':443';


const opsys = process.platform;
if (opsys == "darwin") {    // Mac
    // TEST
    const homedir = require('os').homedir();
    DBS_CONFIG_PATH = `${homedir}/work/LoadBalancer/nginx/dbs_config.xml`;
    NGINX_CONF_PATH = `${homedir}/work/LoadBalancer/nginx/fms_LB_HTTPS.template.conf`;
    NGINX_CONF_OUTPUT_PATH =  `${homedir}/work/LoadBalancer/nginx/fms_LB_HTTPS.user.conf`;
    console.log( "We don't support this in Mac! TEST" );
} 
else if (opsys == "win32" || opsys == "win64") {
    console.log( "We don't support this in Windows!" );
    return 1;
} else if (opsys == "linux") {
    console.log( "Generating nginx config for FMS using nginx load balancer." );
}


if (!fs.existsSync(DBS_CONFIG_PATH)) {
    console.log ( "File " + DBS_CONFIG_PATH + " doesn't exist!");
    return 1;
}

const dbsConfigFile = fs.readFileSync(DBS_CONFIG_PATH, 'utf-8');

// parse xml file to json
const jsonData = JSON.parse(convert.xml2json(dbsConfigFile, {compact: true, spaces: 2}));

const prefData = 
    jsonData.dbsconfig.keys
    .find( x => x._attributes.name === "Preferences" );

const webCompanionData = 
    jsonData.dbsconfig.keys
    .find( x => x._attributes.name === "WebCompanion" );

// check how many servers (primary and secondary) 
console.log ("Total servers (primary and secondary) count= " + webCompanionData.keys.length);

if (webCompanionData.keys.length < 1) {
    console.log( "There is no server config on " + DBS_CONFIG_PATH + "!");
    return 1;
}
const publidIpObj = prefData.key.find( x => x._attributes.name === "PublicIPAddress" );
let primaryServer = publidIpObj ? publidIpObj._text: undefined;

let secondaryServers = [];

for(let i = 0; i < webCompanionData.keys.length; i++)
{
    const wcServer = webCompanionData.keys[i];

    // assume the first key is always primary server
    if (wcServer._attributes.name == "1") {
        continue;
    }

    const wcServerId =  wcServer.key.find( x => x._attributes.name === "Identifier" );
    secondaryServers.push(wcServerId?._text);
}

console.log( "Primary server = " + primaryServer );
console.log( "Secondary servers = " + secondaryServers);

// Read config file
if (!fs.existsSync(NGINX_CONF_PATH)) {
    console.log ( "File " + NGINX_CONF_PATH + " doesn't exist!");
    return 1;
}

NginxConfFile.create(NGINX_CONF_PATH, function (err, conf) {
    if (err || !conf) {
        console.log(err);
        return;
    }

    const onFlushed = () => {
        console.log('finished writing to disk');
    };

    conf.on('flushed', onFlushed);

    conf.die(NGINX_CONF_PATH);

    const upstreamIndex = conf.nginx.http[0].upstream?.length | 0;
    conf.nginx.http?.[0]._add('upstream', 'backend'); // events { }
    secondaryServers.forEach (secondServer => conf.nginx.http[0].upstream?.[upstreamIndex]._add('server', secondServer + HTTPS_PORT));

    if (conf.nginx.http[0].upstream) {
        const upstreamObj = conf.nginx.http[0].upstream[upstreamIndex];
        upstreamObj._comments.push('This will have secondary servers');

        upstreamObj._add('sticky', 'cookie srv_id expires=1h path=/');
        const stickyCount = upstreamObj.sticky.length-1;
        upstreamObj.sticky[stickyCount]._comments.push('using ip hash with nginx open source');
        upstreamObj.sticky[stickyCount]._comments.push('hash $binary_remote_addr consistent;');
        upstreamObj.sticky[stickyCount]._comments.push('using sticky session with nginx plus');
        upstreamObj._add('#server '+ primaryServer + HTTPS_PORT);
    }

    // Add new location blocks in server listening at port 443
    if (conf.nginx.http[0].server && conf.nginx.http[0].server[1]) {

        const server443Obj = conf.nginx.http?.[0].server[1];
        let locationIndex = server443Obj.location?.length | 0 ;
        server443Obj._add('location', '^~ /Streaming/');
        server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/Streaming/');

        locationIndex = server443Obj.location.length;
        server443Obj._add('location', '^~ /Streaming_SSL/');
        server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/Streaming_SSL/');

        locationIndex = server443Obj.location.length;
        server443Obj._add('location', '^~ /fmi/rest/');
        server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/fmi/rest/');

        locationIndex = server443Obj.location.length;
        server443Obj._add('location', '^~ /oauth/');
        server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/oauth/');

        locationIndex = server443Obj.location.length;
        server443Obj._add('location', '/fmi/');
        server443Obj.location[locationIndex]._add('proxy_pass', 'https://backend');
        server443Obj.location[locationIndex]._add('proxy_http_version', '1.1');
        server443Obj.location[locationIndex]._add('proxy_set_header', 'X-Forwarded-Host $host:$server_port');
        server443Obj.location[locationIndex]._add('proxy_set_header', 'X-Forwarded-Server $host');
        server443Obj.location[locationIndex]._add('proxy_set_header', 'X-Forwarded-For $proxy_add_x_forwarded_for');
        server443Obj.location[locationIndex]._add('proxy_set_header', 'Upgrade $http_upgrade');
        server443Obj.location[locationIndex]._add('proxy_set_header', 'Connection  ""');
        server443Obj.location[locationIndex]._add('proxy_set_header', 'Host $host');

        // ideally all FAC should not be included here
        {
            locationIndex = server443Obj.location.length;
            server443Obj._add('location', '/admin-console');
            server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/admin-console');

            locationIndex = server443Obj.location.length;
            server443Obj._add('location', '^~ /fmws/');
            server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/fmws/');

            locationIndex = server443Obj.location.length;
            server443Obj._add('location', '^~ /fmi/admin');
            server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer+'/fmi/admin');

            locationIndex = server443Obj.location.length;
            server443Obj._add('location', '^~ /socket.io/');
            server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer);

            locationIndex = server443Obj.location.length;
            server443Obj._add('location', '@primarybackend');
            server443Obj.location[locationIndex]._add('proxy_pass', 'https://'+ primaryServer);

            locationIndex = server443Obj.location.length;
            server443Obj._add('location', '/');
            server443Obj.location[locationIndex]._add('try_files', '$uri $uri/ @primarybackend');
        }
    }
    const date = new Date();
    conf.nginx._comments.push(" NGINX load balancer template config, generated by FileMaker Tools on "+ date.toDateString());

    //write to a different file
    conf.live(`${NGINX_CONF_OUTPUT_PATH}`);

    // remove old listener
    conf.off('flushed', onFlushed);

    // kill process when done writing to disk
    conf.on('flushed', () => {

        console.log('finished writing to disk, exiting ');
        console.log(`The output file is ${NGINX_CONF_OUTPUT_PATH}`);
        process.exit();
    });

    conf.flush();
    return 0;
});